/**
* \file: ConfigHandle.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \This file is used to read the Key-Value pairs from the Configuration file.
* SPI components like AAUTO and BDCL use ConfigHandle in their EndPoints.
*
* \component: Unified SPI
*
* \author: M. Adachi / ADITJ/SW / madachi@jp.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef USPI_CONFIGHANDLE_H_
#define USPI_CONFIGHANDLE_H_

#include <climits>
#include <string>
#include <set>
#include <map>
#include <vector>

#include <sstream>
#include <algorithm>
#include <stdarg.h>

namespace adit { namespace uspi {

class ConfigHandle
{
public:

    class Validator
    {
    public:
        virtual bool validate(int inValue) const = 0;
        virtual bool validate(std::string inValue) const = 0;
        virtual bool validate(float inValue) const = 0;
        // used as ConfigHandle::novalidator()
        virtual bool validate(void) const = 0;

        virtual ~Validator() {};

    };

    class Range : public Validator
    {
    public:
        Range(int inMin, int inMax);

        bool validate(int inValue) const;
        bool validate(std::string inValue) const;
        bool validate(float inValue) const;
        // used as ConfigHandle::novalidator()
        bool validate(void) const;

     private:
        int min;
        int max;
    };

    class Match : public Validator
    {
    public:
        Match(int inArgNum, ...);
        bool validate(int inValue) const;
        bool validate(std::string inValue) const;
        bool validate(float inValue) const;
        bool validate(void) const;

    private:
        std::set<int> params;
    };

    class MatchStrings : public Validator
    {
    public:
        MatchStrings(int inArgNum, ...);
        bool validate(int inValue) const;
        bool validate(std::string inValue) const;
        bool validate(float inValue) const;
        bool validate(void) const;

    private:
        std::set<std::string> strings;
    };

    std::string getString(const std::string& inKey, bool* outIsValid,
            const ConfigHandle::Validator& inValidator, std::string defaultValue);
    std::string getString(const std::string& inKey, bool* outIsValid,
            const ConfigHandle::Validator& inValidator);
    int getInt(const std::string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator,
            int defaultValue);
    int getInt(const std::string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator);
    float getFloat(const std::string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator,
            float defaultValue);
    float getFloat(const std::string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator);
    int getIntMultiKey(const std::string& inKey, const std::string& inSubkey, bool* outIsValid,
            const ConfigHandle::Validator& inValidator, int defaultValue);
    int getIntMultiKey(const std::string& inKey, const std::string& inSubkey, bool* outIsValid,
            const ConfigHandle::Validator& inValidator);
    std::string getStringMultiKey(const std::string& inKey, const std::string& inSubkey, bool* outIsValid,
            const ConfigHandle::Validator& inValidator, std::string defaultValue);
    std::string getStringMultiKey(const std::string& inKey, const std::string& inSubkey, bool* outIsValid,
            const ConfigHandle::Validator& inValidator);
    float getFloatMultiKey(const std::string& inKey, const std::string& inSubkey, bool* outIsValid,
            const ConfigHandle::Validator& inValidator, float defaultValue);
    float getFloatMultiKey(const std::string& inKey, const std::string& inSubkey, bool* outIsValid,
            const ConfigHandle::Validator& inValidator);

    // set the keys and values into the internal table (map)
    int set(const std::string& inKey, const std::string& inValue);

    ConfigHandle();
    virtual ~ConfigHandle();
private:
    bool find(const std::string& inKey, std::string& outStr);
    bool find(const std::string& inKey, const std::string& inSubKey, std::string& outStr);
    bool findSub(const std::string& inSrcString, const std::string& inFindString, std::string& outDstString);

    std::map<std::string, std::string> parseMap(const std::string& inValue, bool inLowerCase);
    std::vector<std::string> splitString(const std::string& inValue, const char inSeparators[]);
    std::string trimString(const std::string& inValue);


    typedef std::map<std::string, std::string> configTable;
    configTable mConfiguration;
};

} } /* namespace adit { namespace uspi{ */




#endif /* USPI_CONFIGHANDLE_H_ */
